<?php

namespace Modules\Service\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

use Yajra\DataTables\Facades\DataTables;
use Kalnoy\Nestedset\NestedSet;
use Modules\Frontend\Entities\Content;
use Modules\Mwz\Http\Controllers\MwzController;
use Modules\Service\Entities\Services;

class ServiceAdminController extends Controller
{
    /**
     * Function : __construct check admin login
     * Dev : pop
     * Update Date : 14 Jul 2021
     * @param Get
     * @return if not login redirect to /admin
     */
    public function __construct()
    {
        $this->middleware('auth:admin');
    }


    /**
     * Function : service index
     * Dev : pop
     * Update Date : 14 Jul 2021
     * @param Get
     * @return index.blade view
     */
    public function index()
    {

        return view('service::index');
    }


    /**
     * Function : service header
     * Dev : pop
     * Update Date : 14 Jul 2021
     * @param Get
     * @return form_header.blade view
     */
    public function header_service()
    {
        $service_header = Content::find(1);
        $service_header->content_data_th = mwz_getTextString($service_header->content_data_th);
        $service_header->content_data_en = mwz_getTextString($service_header->content_data_en);

        return view('service::form_header', ['service_header' => $service_header]);
    }

    /**
     * Function : service header_save
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param POST
     * @return json response status
     */
    public function header_save(Request $request)
    {
        $attributes = [
            "content_data_th" => mwz_setTextString($request->get('service_header_th')),
            "content_data_en" => mwz_setTextString($request->get('service_header_en'))
        ];

        $Service = Content::where('id', 1)->update($attributes);
        $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกข้อมูลสำเร็จ'];

        return response()->json($resp);
    }


    /**
     * Function : service datatable ajax response
     * Dev : pop
     * Update Date : 14 Jul 2021
     * @param Get
     * @return json of service
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {

            //init datatable
            $dt_name_column = array('sequence', 'image', 'service_name_th', 'updated_at');
            $dt_order_column = $request->get('order')[0]['column'];
            $dt_order_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];

            // count all services
            $dt_total = services::count();

            // create services object
            $o_service = new services;

            // add search query if have search from datable
            if (!empty($dt_search)) {
                $o_service->where('sequence', 'like', "%" . $dt_search . "%")
                    ->where('service_name_th', 'like', "%" . $dt_search . "%");
            }

            // set query order & limit from datatable
            $o_service->orderBy($dt_name_column[$dt_order_column], $dt_order_dir)
                ->offset($dt_start)
                ->limit($dt_length);

            // query services
            $services = $o_service->get();

            // prepare datatable for response
            $tables = Datatables::of($services)
                ->addIndexColumn()
                ->setRowId('sequence')
                ->setRowClass('service_row')
                ->setTotalRecords($dt_total)
                ->editColumn('updated_at', function ($record) {
                    return $record->updated_at->format('Y-m-d H:i:s');
                })
                ->editColumn('image', function ($record) {
                    if ($record->image_service != '' && CheckFileInServer($record->image_service)) {
                        $img = '<img src="' . $record->image_service . '" class="w-7" />';
                    } else {
                        $img = '<img src="/storage/no_image/1920-940.png" class="w-7" />';
                    }
                    return $img;
                })
                ->addColumn('action', function ($record) {
                    $action_btn = '<div class="btn-list">';

                    if ($record->status == 1) {
                        $action_btn .= '<a onclick="setUpdateStatus(' . $record->id . ',0)" href="javascript:void(0);" class="btn btn-outline-success"><i class="fa fa-check"></i></a></a>';
                    } else {
                        $action_btn .=  '<a onclick="setUpdateStatus(' . $record->id . ',1)" href="javascript:void(0);"  class="btn btn-outline-warning"><i class="fa fa-times"></i></a></a>';
                    }

                    $action_btn .= '<a href="' . route('admin.service.service.edit', $record->id) . '" class="btn btn-outline-primary"><i class="fa fa-pencil"></i></a></a>';
                    $action_btn .= '<a onclick="setDelete(' . $record->id . ')" href="javascript:void(0);" class="btn btn-outline-danger"><i class="fa fa-trash"></i></a></a>';
                    $action_btn .= '</div>';

                    return $action_btn;
                })
                ->escapeColumns([]);

            // response datatable json
            return $tables->make(true);
        }
    }

    /**
     * Function : add service form
     * Dev : pop
     * Update Date : 14 Jul 2021
     * @param GET
     * @return service form view
     */
    public function form($id = 0)
    {
        $service = [];

        if (!empty($id)) {
            $service = Services::find($id);
            $service->description_th = mwz_getTextString($service->description_th);
            $service->description_en = mwz_getTextString($service->description_en);
            $service->detail_th = mwz_getTextString($service->detail_th);
            $service->detail_en = mwz_getTextString($service->detail_en);
        }


        return view('service::form', ['service' => $service]);
    }


    /**
     * Function : service save
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param POST
     * @return json response status
     */
    public function save(Request $request)
    {

        if (empty($request->get('service_name_th'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุหัวข้อภาษาไทย!', 'focus' => 'service_name_th'];
            return response()->json($resp);
        }
        if (empty($request->get('service_name_en'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุหัวข้อภาษาอังกฤษ!', 'focus' => 'service_name_en'];
            return response()->json($resp);
        }
        if (empty($request->get('detail_th'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุคำอธิบายภาษาไทย!', 'focus' => 'detail_th'];
            return response()->json($resp);
        }
        if (empty($request->get('detail_en'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุคำอธิบายภาษาอังกฤษ!', 'focus' => 'detail_en'];
            return response()->json($resp);
        }
        if (empty($request->get('description_th'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุรายละเอียดภาษาไทย!', 'focus' => 'description_th'];
            return response()->json($resp);
        }
        if (empty($request->get('description_en'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุรายละเอียดภาษาอังกฤษ!', 'focus' => 'description_en'];
            return response()->json($resp);
        }
        if (empty($request->get('sequence'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุลำดับการแสดงผล!', 'focus' => 'sequence'];
            return response()->json($resp);
        }

        //validate post data
        $validator = Validator::make($request->all(), [
            'service_name_th' => 'required',
            'service_name_en' => 'required',
            'detail_th' => 'required',
            'detail_en' => 'required',
            'description_th' => 'required',
            'description_en' => 'required',
            'sequence' => 'required',
            'status' => 'required'
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors();
            $resp = ['success' => 0, 'code' => 301, 'msg' => 'error', 'error' => $errors];
            return response()->json($resp);
        }

        $attributes = [
            "service_name_th" => $request->get('service_name_th'),
            "service_name_en" => $request->get('service_name_en'),
            "detail_th" => mwz_setTextString($request->get('detail_th')),
            "detail_en" => mwz_setTextString($request->get('detail_en')),
            "description_th" => mwz_setTextString($request->get('description_th')),
            "description_en" => mwz_setTextString($request->get('description_en')),
            "sequence" => $request->get('sequence'),
            "status" => $request->get('status')
        ];

        if ($request->hasFile('image_service')) {
            $image = $request->file('image_service');
            $new_filename = time() . "." . $image->extension();
            $path = $image->storeAs(
                'public/service',
                $new_filename
            );
            $attributes['image_service'] = Storage::url($path);
        }
        if ($request->hasFile('image_home')) {
            $image = $request->file('image_home');
            $new_filename = time() . "." . $image->extension();
            $path = $image->storeAs(
                'public/service',
                $new_filename
            );
            $attributes['image_home'] = Storage::url($path);
        }

        if ($request->file('file')) {
            $file = $request->file('file');
            $filename = time() . '.' . $request->file('file')->extension();
            $path = $file->storeAs(
                'public/service',
                $filename
            );
            $attributes['file'] = Storage::url($path);
        }

        if (!empty($request->get('id'))) {
            Services::where('id', $request->get('id'))->update($attributes);
            $resp = ['success' => 1, 'code' => 200, 'msg' => 'อัปเดตข้อมูลสำเร็จ'];
        } else {
            $service = Services::create($attributes);
            $service->save();
            $resp = ['success' => 1, 'code' => 201, 'msg' => 'บันทึกข้อมูลสำเร็จ'];
        }

        return response()->json($resp);
    }

    /**
     * Function : update service status
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param POST
     * @return json of update status
     */
    public function set_status(Request $request)
    {
        if ($request->ajax()) {
            $id = $request->get('id');
            $status = $request->get('status');

            $service = Services::find($id);
            $service->status = $status;

            if ($service->save()) {
                $resp = ['success' => 1, 'code' => 200, 'msg' => 'อัปเดตสถานะสำเร็จ'];
            } else {
                $resp = ['success' => 0, 'code' => 500, 'msg' => 'อัปเดตสถานะไม่สำเร็จ!'];
            }

            return response()->json($resp);
        }
    }

    /**
     * Function : delete service
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param POST
     * @return json of delete status
     */
    public function set_delete(Request $request)
    {
        if ($request->ajax()) {
            $id = $request->get('id');
            $service = Services::find($id);

            if ($service->delete()) {
                $resp = ['success' => 1, 'code' => 200, 'msg' => 'ลบข้อมูลบริการสำเร็จ'];
            } else {
                $resp = ['success' => 0, 'code' => 500, 'msg' => 'ลบข้อมูลบริการไม่สำเร็จ!'];
            }

            return response()->json($resp);
        }
    }

    /**
     * Function : delete image
     * Dev : Ta
     * Update Date : 25 Aug 2021
     * @param POST
     * @return json of response
     */
    public function delete_image(Request $request)
    {
        if ($request->ajax()) {
            $id = $request->get('id');
            $type = $request->get('type');

            if ($type == 1) {
                $service = Services::find($id);
                $service->image_service = '';

                if ($service->save()) {
                    $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกการเปลี่ยนแปลงสำเร็จ'];
                } else {
                    $resp = ['success' => 0, 'code' => 500, 'msg' => 'เกิดข้อผิดพลาด โปรดลองใหม่อีกครั้ง!'];
                }
            } else {
                $service = Services::find($id);
                $service->image_home = '';

                if ($service->save()) {
                    $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกการเปลี่ยนแปลงสำเร็จ'];
                } else {
                    $resp = ['success' => 0, 'code' => 500, 'msg' => 'เกิดข้อผิดพลาด โปรดลองใหม่อีกครั้ง!'];
                }
            }

            return response()->json($resp);
        }
    }
}
