<?php

namespace Modules\Blog\Http\Controllers;

use App\Http\Livewire\Blog;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

use Yajra\DataTables\Facades\DataTables;
use Kalnoy\Nestedset\NestedSet;

use Modules\Mwz\Http\Controllers\MwzController;
use Modules\Blog\Entities\BlogCategories;
use Modules\Blog\Entities\Blogs;

class BlogAdminController extends Controller
{
    /**
     * Function : __construct check admin login
     * Dev : pop
     * Update Date : 14 Jul 2021
     * @param Get
     * @return if not login redirect to /admin
     */
    public function __construct()
    {
        $this->middleware('auth:admin');
    }

    /**
     * Function : blog index
     * Dev : pop
     * Update Date : 14 Jul 2021
     * @param Get
     * @return index.blade view
     */
    public function index()
    {

        return view('blog::index');
    }

    /**
     * Function : blog datatable ajax response
     * Dev : pop
     * Update Date : 14 Jul 2021
     * @param Get
     * @return json of blog
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {

            //init datatable
            $dt_name_column = array('_lft', 'sequence', 'image', 'name_th', 'updated_at');
            $dt_order_column = $request->get('order')[0]['column'];
            $dt_order_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];

            // count all blogs
            $dt_total = Blogs::count();

            // create blogs object
            $o_blog = new Blogs;

            // add search query if have search from datable
            if (!empty($dt_search)) {
                $o_blog->where('name_th', 'like', "%" . $dt_search . "%")
                    ->where('sequence', 'like', "%" . $dt_search . "%");
            }

            // set query order & limit from datatable
            $o_blog->orderBy($dt_name_column[$dt_order_column], $dt_order_dir)
                ->offset($dt_start)
                ->limit($dt_length);

            // query blogs
            $blogs = $o_blog->join('blog_categories', 'blogs.category_id', '=', 'blog_categories.id')
                ->select('blogs.id', 'blogs.name_th', 'blogs.sequence', 'blogs.updated_at', 'blogs.image', 'blogs.status', 'blog_categories.name_th as cat_name')
                ->get();

            // prepare datatable for response
            $tables = Datatables::of($blogs)
                ->addIndexColumn()
                ->setRowId('sequence')
                ->setRowClass('blog_row')
                ->setTotalRecords($dt_total)
                ->editColumn('updated_at', function ($record) {
                    return $record->updated_at->format('Y-m-d H:i:s');
                })
                ->editColumn('image', function ($record) {
                    if ($record->image != '' && CheckFileInServer($record->image)) {
                        $img = '<img src="' . $record->image . '" class="w-7" />';
                    } else {
                        $img = '<img src="/storage/no_image/1920-940.png" class="w-7" />';
                    }
                    return $img;
                })
                ->addColumn('action', function ($record) {
                    $action_btn = '<div class="btn-list">';

                    if ($record->status == 1) {
                        $action_btn .= '<a onclick="setUpdateStatus(' . $record->id . ',0)" href="javascript:void(0);" class="btn btn-outline-success"><i class="fa fa-check"></i></a></a>';
                    } else {
                        $action_btn .=  '<a onclick="setUpdateStatus(' . $record->id . ',1)" href="javascript:void(0);"  class="btn btn-outline-warning"><i class="fa fa-times"></i></a></a>';
                    }

                    $action_btn .= '<a href="' . route('admin.blog.blog.edit', $record->id) . '" class="btn btn-outline-primary"><i class="fa fa-pencil"></i></a></a>';
                    $action_btn .= '<a onclick="setDelete(' . $record->id . ')" href="javascript:void(0);" class="btn btn-outline-danger"><i class="fa fa-trash"></i></a></a>';
                    $action_btn .= '</div>';

                    return $action_btn;
                })
                ->escapeColumns([]);

            // response datatable json
            return $tables->make(true);
        }
    }

    /**
     * Function : add blog form
     * Dev : pop
     * Update Date : 14 Jul 2021
     * @param GET
     * @return blog form view
     */
    public function form($id = 0)
    {
        $blog = [];

        if (!empty($id)) {
            $blog = Blogs::find($id);
            $blog->description_th = mwz_getTextString($blog->description_th);
            $blog->description_en = mwz_getTextString($blog->description_en);
        }

        $categories = BlogCategories::all();

        return view('blog::form', ['categories' => $categories, 'blog' => $blog]);
    }

    /**
     * Function :  blog save
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param POST
     * @return json response status
     */
    public function save(Request $request)
    {
        //Check input is null
        if (empty($request->get('name_th'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุหัวข้อบล็อกภาษาไทย!', 'focus' => 'name_th'];
            return response()->json($resp);
        }
        if (empty($request->get('name_en'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุหัวข้อบล็อกภาษาอังกฤษ!', 'focus' => 'name_en'];
            return response()->json($resp);
        }
        if (empty($request->get('description_th'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุรายละเอียดภาษาไทย!', 'focus' => 'description_th'];
            return response()->json($resp);
        }
        if (empty($request->get('description_en'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุรายละเอียดภาษาอังกฤษ!', 'focus' => 'description_en'];
            return response()->json($resp);
        }
        if (empty($request->get('category_id'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดเลือกหมวดหมู่!', 'focus' => 'category_id'];
            return response()->json($resp);
        }
        if (empty($request->get('publish_at'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุวันที่สร้าง!', 'focus' => 'publish_at'];
            return response()->json($resp);
        }
        if (empty($request->get('sequence'))) {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุลำดับการแสดงผล!', 'focus' => 'sequence'];
            return response()->json($resp);
        }


        //validate post data
        $validator = Validator::make($request->all(), [
            'id' => 'integer',
            'category_id' => 'required|integer',
            'name_th' => 'required',
            'name_en' => 'required',
            'description_th' => 'required',
            'description_en' => 'required',
            'sequence' => 'required',
            'status' => 'required',
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors();
            $resp = ['success' => 0, 'code' => 301, 'msg' => 'เกิดข้อผิดพลาด โปรดลองใหม่อีกครั้ง!', 'error' => $errors];
            return response()->json($resp);
        }

        $publish_at = date('Y-m-d', strtotime($request->publish_at));

        $attributes = [
            "category_id" => $request->get('category_id'),
            "name_th" => $request->get('name_th'),
            "name_en" => $request->get('name_en'),
            "description_th" => mwz_setTextString($request->get('description_th')),
            "description_en" => mwz_setTextString($request->get('description_en')),
            "link" => $request->get('link'),
            "author" => auth()->id(),
            "sequence" => $request->get('sequence'),
            "status" => $request->get('status'),
            "publish_at" => $publish_at,
            "tags" => $request->get('tags')
        ];


        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $new_filename = time() . "." . $image->extension();
            $path = $image->storeAs(
                'public/blog',
                $new_filename
            );
            $attributes['image'] = Storage::url($path);
        }

        if ($request->file('file')) {
            $file = $request->file('file');
            $filename = time() . '.' . $request->file('file')->extension();
            $path = $file->storeAs(
                'public/blog',
                $filename
            );
            $attributes['file'] = Storage::url($path);
        }

        if (!empty($request->get('id'))) {
            $Blog = Blogs::where('id', $request->get('id'))->update($attributes);
            $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกการเปลี่ยนแปลงสำเร็จ'];
        } else {
            $Blog = Blogs::create($attributes);
            $resp = ['success' => 1, 'code' => 201, 'msg' => 'เพิ่มรายการใหม่สำเร็จ'];
        }

        return response()->json($resp);
    }

    /**
     * Function : update blog status
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param POST
     * @return json of update status
     */
    public function set_status(Request $request)
    {
        if ($request->ajax()) {
            $id = $request->get('id');
            $status = $request->get('status');

            $blog = Blogs::find($id);
            $blog->status = $status;

            if ($blog->save()) {
                $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกการเปลี่ยนแปลงสำเร็จ'];
            } else {
                $resp = ['success' => 0, 'code' => 500, 'msg' => 'เกิดข้อผิดพลาด โปรดลองใหม่อีกครั้ง!'];
            }

            return response()->json($resp);
        }
    }

    /**
     * Function : delete blog
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param POST
     * @return json of delete status
     */
    public function set_delete(Request $request)
    {
        if ($request->ajax()) {
            $id = $request->get('id');
            $blog = Blogs::find($id);

            if ($blog->delete()) {
                $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกการเปลี่ยนแปลงสำเร็จ'];
            } else {
                $resp = ['success' => 0, 'code' => 500, 'msg' => 'เกิดข้อผิดพลาด โปรดลองใหม่อีกครั้ง!'];
            }

            return response()->json($resp);
        }
    }

    /**
     * Function : blog category index
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param Get
     * @return category.blade view
     */
    public function category()
    {
        return view('blog::category');
    }

    /**
     * Function : blog category datatable ajax response
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param Get
     * @return json of blog category
     */
    public function category_datatable_ajax(Request $request)
    {
        if ($request->ajax()) {

            //init datatable
            $dt_name_column = array('_lft', 'id', 'image', 'name_th', 'status', 'updated_at');
            $dt_order_column = $request->get('order')[0]['column'];
            $dt_order_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];

            // count all category
            $dt_total = BlogCategories::count();
            // create categroy object
            $o_categories = new BlogCategories;

            // add search query if have search from datable
            if (!empty($dt_search)) {
                $o_categories->where('name_th', 'like', "%" . $dt_search . "%")
                    ->where('id', 'like', "%" . $dt_search . "%");
            }

            // set query order & limit from datatable
            $o_categories->orderBy($dt_name_column[$dt_order_column], $dt_order_dir)
                ->offset($dt_start)
                ->limit($dt_length);

            // query category as tree resule
            $categories = $o_categories->get(['id', 'image', 'name_th', 'status', 'updated_at'])->toTree(); //,'_lft','_rgt','parent_id'

            // set flat category from helper function
            $categories = mwz_setFlatCategory($categories);

            // prepare datatable for resonse
            $tables = Datatables::of($categories)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('master_row')
                ->setTotalRecords($dt_total)
                ->editColumn('updated_at', function ($record) {
                    return $record->updated_at->format('Y-m-d H:i:s');
                })
                ->editColumn('image', function ($record) {
                    if ($record->image != '' && CheckFileInServer($record->image)) {
                        $img = '<img src="' . $record->image . '" class="w-7" />';
                    } else {
                        $img = '<img src="/storage/no_image/420-260.png" class="w-7" />';
                    }
                    return $img;
                })
                ->addColumn('action', function ($record) {
                    $action_btn = '<div class="btn-list">';

                    if ($record->status == 1) {
                        $action_btn .= '<a onclick="setUpdateCategoryStatus(' . $record->id . ',0)" href="javascript:void(0);" class="btn btn-outline-success"><i class="fa fa-check"></i></a></a>';
                    } else {
                        $action_btn .=  '<a onclick="setUpdateCategoryStatus(' . $record->id . ',1)" href="javascript:void(0);"  class="btn btn-outline-warning"><i class="fa fa-times"></i></a></a>';
                    }

                    $action_btn .= '<a href="' . route('admin.blog.category.edit', $record->id) . '" class="btn btn-outline-primary"><i class="fa fa-pencil"></i></a></a>';

                    $action = '</div>';

                    return $action_btn;
                })
                ->escapeColumns([]);

            // response datatable json
            return $tables->make(true);
        }
    }

    /**
     * Function : add blog category form
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param GET
     * @return category form view
     */
    public function category_form($category_id = 0)
    {
        $category = [];
        if (!empty($category_id)) {
            $category = BlogCategories::find($category_id);
            $category->description_th = mwz_getTextString($category->description_th);
            $category->description_en = mwz_getTextString($category->description_en);
        }

        $parents = BlogCategories::all()->totree();
        $parents = mwz_setFlatCategory($parents);


        return view('blog::category_form', ['parents' => $parents, 'category' => $category]);
    }

    /**
     * Function :  save blog category
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param POST
     * @return json of response
     */
    public function category_save(Request $request)
    {
        if ($request->get('name_th') == "") {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุชื่อหมวดหมู่ภาษาไทย!', 'focus' => 'name_th'];
            return response()->json($resp);
        }

        if ($request->get('name_en') == "") {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุชื่อหมวดหมู่ภาษาอังกฤษ!', 'focus' => 'name_en'];
            return response()->json($resp);
        }

        if ($request->get('description_th') == "") {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุรายละเอียดภาษาไทย!', 'focus' => 'description_th'];
            return response()->json($resp);
        }

        if ($request->get('description_en') == "") {
            $resp = ['error' => 0, 'code' => 301, 'msg' => 'โปรดระบุรายละเอียดภาษาอังกฤษ!', 'focus' => 'description_en'];
            return response()->json($resp);
        }

        //validate post data
        $validator = Validator::make($request->all(), [
            'id' => 'integer',
            'name_th' => 'required|max:500',
            'name_en' => 'required|max:500',
            'description_th' => 'max:500',
            'description_en' => 'max:500',
            'status' => 'required|integer'

        ]);

        if ($validator->fails()) {
            $errors = $validator->errors();
            $resp = ['success' => 0, 'code' => 0, 'msg' => 'เกิดข้อผิดพลาด โปรดลองใหม่อีกครั้ง!', 'error' => $errors];
            return response()->json($resp);
        }

        $now = DB::raw('NOW()');
        $attributes = [
            "name_th" => $request->get('name_th'),
            "name_en" => $request->get('name_en'),
            "description_th" => mwz_setTextString($request->get('description_th')),
            "description_en" => mwz_setTextString($request->get('description_en')),
            "status" => $request->get('status')
        ];

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $new_filename = time() . "." . $image->extension();
            $path = $image->storeAs(
                'public/blog',
                $new_filename
            );
            $attributes['image'] = Storage::url($path);
        }


        if (!empty($request->get('id'))) {
            $blog_category = BlogCategories::where('id', $request->get('id'))->update($attributes);
            $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกการเปลี่ยนแปลงสำเร็จ'];
        } else {
            $blog_category = BlogCategories::create($attributes);
            $blog_category->save();
            $resp = ['success' => 1, 'code' => 201, 'msg' => 'เพิ่มรายการใหม่สำเร็จ'];
        }

        return response()->json($resp);
    }


    /**
     * Function : update blog category status
     * Dev : pop
     * Update Date : 11 Jul 2021
     * @param POST
     * @return json of update status
     */
    public function set_category_status(Request $request)
    {
        if ($request->ajax()) {
            $category_id = $request->get('category_id');
            $status = $request->get('status');

            $category = BlogCategories::find($category_id);
            $category->status = $status;

            if ($category->save()) {
                $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกการเปลี่ยนแปลงสำเร็จ'];
            } else {
                $resp = ['success' => 0, 'code' => 500, 'msg' => 'เกิดข้อผิดพลาด โปรดลองใหม่อีกครั้ง!'];
            }

            return response()->json($resp);
        }
    }

    /**
     * Function : delete image
     * Dev : Ta
     * Update Date : 25 Aug 2021
     * @param POST
     * @return json of response
     */
    public function delete_image(Request $request)
    {
        if ($request->ajax()) {
            $id = $request->get('id');
            $type = $request->get('type');

            if ($type == 1) {
                $blog = BlogCategories::find($id);
                $blog->image = '';

                if ($blog->save()) {
                    $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกการเปลี่ยนแปลงสำเร็จ'];
                } else {
                    $resp = ['success' => 0, 'code' => 500, 'msg' => 'เกิดข้อผิดพลาด โปรดลองใหม่อีกครั้ง!'];
                }
            } else {
                $blog = Blogs::find($id);
                $blog->image = '';

                if ($blog->save()) {
                    $resp = ['success' => 1, 'code' => 200, 'msg' => 'บันทึกการเปลี่ยนแปลงสำเร็จ'];
                } else {
                    $resp = ['success' => 0, 'code' => 500, 'msg' => 'เกิดข้อผิดพลาด โปรดลองใหม่อีกครั้ง!'];
                }
            }

            return response()->json($resp);
        }
    }
}
